package com.devstudio.myshop.service;

import com.devstudio.myshop.model.Template;
import com.devstudio.myshop.model.TemplateCategory;
import com.devstudio.myshop.model.Service;
import com.devstudio.myshop.model.ProjectCase;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

@org.springframework.stereotype.Service
@Component
public class DataService {
    
    private final ObjectMapper objectMapper = new ObjectMapper();
    private List<TemplateCategory> templateCategories = new ArrayList<>();
    private List<Template> templates = new ArrayList<>();
    private List<Service> services = new ArrayList<>();
    private List<ProjectCase> projectCases = new ArrayList<>();
    
    @PostConstruct
    public void init() {
        System.out.println("Initializing DataService...");
        try {
            loadTemplateCategories();
            System.out.println("Loaded " + templateCategories.size() + " template categories");
            
            loadTemplates();
            System.out.println("Loaded " + templates.size() + " templates");
            
            loadServices();
            System.out.println("Loaded " + services.size() + " services");
            
            loadProjectCases();
            System.out.println("Loaded " + projectCases.size() + " project cases");
        } catch (Exception e) {
            System.err.println("Error loading data: " + e.getMessage());
            e.printStackTrace();
        }
        System.out.println("DataService initialization completed.");
    }
    
    private void loadTemplateCategories() throws IOException {
        try (InputStream is = getClass().getResourceAsStream("/data/template-categories.json")) {
            if (is != null) {
                templateCategories = objectMapper.readValue(is, new TypeReference<List<TemplateCategory>>() {});
            }
        }
    }
    
    private void loadTemplates() throws IOException {
        try (InputStream is = getClass().getResourceAsStream("/data/templates.json")) {
            if (is != null) {
                templates = objectMapper.readValue(is, new TypeReference<List<Template>>() {});
            }
        }
    }
    
    private void loadServices() throws IOException {
        try (InputStream is = getClass().getResourceAsStream("/data/services.json")) {
            if (is != null) {
                services = objectMapper.readValue(is, new TypeReference<List<Service>>() {});
            }
        }
    }
    
    private void loadProjectCases() throws IOException {
        try (InputStream is = getClass().getResourceAsStream("/data/project-cases.json")) {
            if (is != null) {
                projectCases = objectMapper.readValue(is, new TypeReference<List<ProjectCase>>() {});
            }
        }
    }
    
    public List<TemplateCategory> getTemplateCategories() {
        return templateCategories;
    }
    
    public TemplateCategory getTemplateCategoryById(String id) {
        return templateCategories.stream()
                .filter(category -> category.getId().equals(id))
                .findFirst()
                .orElse(null);
    }
    
    public List<Template> getTemplates() {
        return templates;
    }
    
    public List<Template> getTemplatesByCategory(String categoryId) {
        return templates.stream()
                .filter(template -> template.getCategoryId().equals(categoryId))
                .collect(Collectors.toList());
    }
    
    public Template getTemplateById(String id) {
        return templates.stream()
                .filter(template -> template.getId().equals(id))
                .findFirst()
                .orElse(null);
    }
    
    public List<Template> getPopularTemplates() {
        return templates.stream()
                .filter(Template::isPopular)
                .collect(Collectors.toList());
    }
    
    public List<Service> getServices() {
        return services;
    }
    
    public Service getServiceById(String id) {
        return services.stream()
                .filter(service -> service.getId().equals(id))
                .findFirst()
                .orElse(null);
    }
    
    public List<ProjectCase> getProjectCases() {
        return projectCases;
    }
    
    public List<ProjectCase> getProjectCasesByServiceType(String serviceType) {
        return projectCases.stream()
                .filter(projectCase -> projectCase.getServiceType().equals(serviceType))
                .collect(Collectors.toList());
    }
    
    public ProjectCase getProjectCaseById(String id) {
        return projectCases.stream()
                .filter(projectCase -> projectCase.getId().equals(id))
                .findFirst()
                .orElse(null);
    }
    
    public List<ProjectCase> getFeaturedCases() {
        return projectCases.stream()
                .limit(6)
                .collect(Collectors.toList());
    }
}