package com.devstudio.myshop.service;

import com.devstudio.myshop.model.ContactForm;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;

@Service
public class ContactService {
    
    private final String CONTACT_DATA_DIR = "contact-submissions";
    private final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    
    public String saveContactForm(ContactForm contactForm) {
        try {
            // Ensure directory exists
            Path contactDir = Paths.get(CONTACT_DATA_DIR);
            if (!Files.exists(contactDir)) {
                Files.createDirectories(contactDir);
            }
            
            // Create a filename with timestamp
            String timestamp = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMdd-HHmmss"));
            String filename = "contact-" + timestamp + ".txt";
            Path filePath = contactDir.resolve(filename);
            
            // Format contact information
            StringBuilder content = new StringBuilder();
            content.append("========== 提交时间: ").append(LocalDateTime.now().format(DATE_FORMATTER)).append(" ==========\n");
            content.append("姓名: ").append(contactForm.getName()).append("\n");
            content.append("邮箱: ").append(contactForm.getEmail()).append("\n");
            content.append("电话: ").append(contactForm.getPhone()).append("\n");
            if (contactForm.getServiceType() != null && !contactForm.getServiceType().isEmpty()) {
                content.append("感兴趣的服务: ").append(contactForm.getServiceType()).append("\n");
            }
            content.append("需求描述: ").append(contactForm.getMessage()).append("\n");
            content.append("==================================================\n\n");
            
            // Write to file with UTF-8 encoding
            Files.write(filePath, content.toString().getBytes(StandardCharsets.UTF_8), 
                       StandardOpenOption.CREATE, StandardOpenOption.WRITE);
            
            return "success";
        } catch (IOException e) {
            System.err.println("Error saving contact form: " + e.getMessage());
            return "error";
        }
    }
    
    public java.util.List<ContactForm> getAllContactSubmissions() {
        java.util.List<ContactForm> submissions = new java.util.ArrayList<>();
        
        try {
            Path contactDir = Paths.get(CONTACT_DATA_DIR);
            if (!Files.exists(contactDir)) {
                return submissions;
            }
            
            // Read all files in directory, sorted by modification time (newest first)
            java.util.List<Path> files = Files.list(contactDir)
                .filter(path -> path.getFileName().toString().startsWith("contact-") && 
                               path.getFileName().toString().endsWith(".txt"))
                .sorted((p1, p2) -> {
                    try {
                        return Files.getLastModifiedTime(p2).compareTo(Files.getLastModifiedTime(p1));
                    } catch (IOException e) {
                        return 0;
                    }
                })
                .collect(java.util.stream.Collectors.toList());
            
            for (Path file : files) {
                try {
                    String content = new String(Files.readAllBytes(file), StandardCharsets.UTF_8);
                    ContactForm contact = parseContactContent(content);
                    if (contact != null) {
                        submissions.add(contact);
                    }
                } catch (IOException e) {
                    System.err.println("Error reading file " + file + ": " + e.getMessage());
                }
            }
        } catch (IOException e) {
            System.err.println("Error reading contact submissions: " + e.getMessage());
        }
        
        return submissions;
    }
    
    private ContactForm parseContactContent(String content) {
        try {
            ContactForm contact = new ContactForm();
            String[] lines = content.split("\\r?\\n");
            
            for (String line : lines) {
                if (line.startsWith("姓名: ")) {
                    contact.setName(line.substring(4).trim());
                } else if (line.startsWith("邮箱: ")) {
                    contact.setEmail(line.substring(4).trim());
                } else if (line.startsWith("电话: ")) {
                    contact.setPhone(line.substring(4).trim());
                } else if (line.startsWith("感兴趣的服务: ")) {
                    contact.setServiceType(line.substring(8).trim());
                } else if (line.startsWith("需求描述: ")) {
                    contact.setMessage(line.substring(6).trim());
                } else if (line.startsWith("========== 提交时间: ")) {
                    // Store submission time in a custom field if needed
                    // For now, we'll store it in the message field temporarily
                    String timestamp = line.substring(13, line.indexOf(" =========="));
                    // We'll need to modify ContactForm to include this field later
                }
            }
            
            return contact;
        } catch (Exception e) {
            System.err.println("Error parsing contact content: " + e.getMessage());
            return null;
        }
    }
}