package com.devstudio.myshop.controller;

import com.devstudio.myshop.model.Template;
import com.devstudio.myshop.model.TemplateCategory;
import com.devstudio.myshop.service.DataService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestParam;

import java.util.List;

@Controller
public class TemplateController {
    
    @Autowired
    private DataService dataService;
    
    @GetMapping("/templates")
    public String templates(
            @RequestParam(name = "category", required = false) String categoryId,
            @RequestParam(name = "search", required = false) String search,
            Model model) {
        
        List<TemplateCategory> categories = dataService.getTemplateCategories();
        model.addAttribute("categories", categories);
        
        List<Template> templates;
        if (categoryId != null && !categoryId.isEmpty()) {
            templates = dataService.getTemplatesByCategory(categoryId);
            model.addAttribute("selectedCategory", categoryId);
        } else {
            templates = dataService.getTemplates();
        }
        
        // Filter by search term if provided
        if (search != null && !search.isEmpty()) {
            String searchTerm = search.toLowerCase();
            templates = templates.stream()
                    .filter(template -> 
                        template.getName().toLowerCase().contains(searchTerm) || 
                        template.getDescription().toLowerCase().contains(searchTerm))
                    .collect(java.util.stream.Collectors.toList());
            model.addAttribute("searchTerm", search);
        }
        
        model.addAttribute("templates", templates);
        
        return "templates";
    }
    
    @GetMapping("/templates/{id}")
    public String templateDetail(@PathVariable String id, Model model) {
        Template template = dataService.getTemplateById(id);
        if (template == null) {
            return "redirect:/templates";
        }
        
        model.addAttribute("template", template);
        
        TemplateCategory category = dataService.getTemplateCategoryById(template.getCategoryId());
        model.addAttribute("category", category);
        
        // Get related templates from the same category
        List<Template> relatedTemplates = dataService.getTemplatesByCategory(template.getCategoryId());
        // Remove current template from related templates
        relatedTemplates.removeIf(t -> t.getId().equals(id));
        // Limit to 3 related templates
        if (relatedTemplates.size() > 3) {
            relatedTemplates = relatedTemplates.subList(0, 3);
        }
        model.addAttribute("relatedTemplates", relatedTemplates);
        
        return "template-detail";
    }
    
    @GetMapping("/preview/{id}")
    public String previewTemplate(@PathVariable String id, Model model) {
        Template template = dataService.getTemplateById(id);
        if (template == null) {
            return "redirect:/templates";
        }
        
        model.addAttribute("template", template);
        
        return "template-preview";
    }
}